;;; coq-db.el --- coq keywords database utility functions -*- coding: utf-8; lexical-binding:t -*-

;; This file is part of Proof General.

;; Portions © Copyright 1994-2012  David Aspinall and University of Edinburgh
;; Portions © Copyright 2003-2018  Free Software Foundation, Inc.
;; Portions © Copyright 2001-2017  Pierre Courtieu
;; Portions © Copyright 2010, 2016  Erik Martin-Dorel
;; Portions © Copyright 2011-2013, 2016-2017  Hendrik Tews
;; Portions © Copyright 2015-2017  Clément Pit-Claudel

;; Author: Pierre Courtieu <courtieu@lri.fr>

;; SPDX-License-Identifier: GPL-3.0-or-later

;;; Commentary:
;;
;; We store all information on keywords (tactics or command) in big
;; tables (ex: `coq-tactics-db') From there we get: menus including
;; "smart" commands, completions for command coq-insert-...
;; abbrev tables and font-lock keyword.
;;

;;; Code:

(eval-when-compile (require 'cl-lib))   ;decf

(require 'diff-mode)

(defconst coq-syntax-db nil
  "Documentation-only variable, for coq keyword databases.
Each element of a keyword database contains the definition of a \"form\", of the
form:

\(MENUNAME ABBREV INSERT STATECH KWREG INSERT-FUN HIDE)

MENUNAME is the name of form (or form variant) as it should appear in menus or
completion lists.

ABBREV is the abbreviation for completion via \\[expand-abbrev].

INSERT is the complete text of the form, which may contain holes denoted by
\"#\" or \"@{xxx}\".

If non-nil the optional STATECH specifies that the command is not state
preserving for coq.

If non-nil the optional KWREG is the regexp to colorize correponding to the
keyword.  ex: \"simple\\\\s-+destruct\" (\\\\s-+ meaning \"one or more spaces\").
*WARNING*: A regexp longer than another one should be put FIRST. For example:

  (\"Module Type\" ... ... t \"Module\\s-+Type\")
  (\"Module\" ... ... t \"Module\")

Is ok because the longer regexp is recognized first.

If non-nil the optional INSERT-FUN is the function to be called when inserting
the form (instead of inserting INSERT, except when using \\[expand-abbrev]). This
allows to write functions asking for more information to assist the user.

If non-nil the optional HIDE specifies that this form should not appear in the
menu but only in interactive completions.

Example of what could be in your emacs init file:

\(defvar coq-user-tactics-db
  \\='(
    (\"mytac\" \"mt\" \"mytac # #\" t \"mytac\")
    (\"myassert by\" \"massb\" \"myassert ( # : # ) by #\" t \"assert\")
    ))

Explanation of the first line: the tactic menu entry mytac, abbreviated by mt,
will insert \"mytac # #\" where #s are holes to fill, and \"mytac\" becomes a
new keyword to colorize.")

(defun coq-insert-from-db (db prompt &optional alwaysjump)
  "Ask for a keyword, with completion on keyword database DB and insert.
Insert corresponding string with holes at point.  If an insertion
function is present for the keyword, call it instead. See
`coq-syntax-db' for DB structure.  If ALWAYSJUMP is non nil, jump to
the first hole even if more than one."
  (let* ((tac (completing-read (concat prompt " (TAB for completion): ")
			       db nil nil))
	 (infos (cddr (assoc tac db)))
	 (s (car infos)) ; completion to insert
	 (f (car-safe (cdr-safe (cdr-safe (cdr infos))))) ; insertion function
	 (pt (point)))
    (if f (funcall f) ; call f if present
      (insert (or s tac)) ; insert completion and indent otherwise
      ;; FIXHOLE
      ;;(holes-replace-string-by-holes-backward-jump pt nil alwaysjump)
      (indent-according-to-mode))))



(defun coq-build-command-from-db (db prompt &optional _preformatquery)
  "Ask for a keyword, with completion on keyword database DB and send to Coq.
See `coq-syntax-db' for DB structure."
  ;; Next invocation of minibuffer (read-string below) will first recursively
  ;; ask for a command in db and expand it with holes. This way the cursor will
  ;; be at the right place.
  (minibuffer-with-setup-hook
      (lambda () (coq-insert-from-db db prompt t))
    ;; I use recursive edition on the minibuffer here, because I want the cursor
    ;; to be moved inside the initial content
    (let ((enable-recursive-minibuffers t)) ; invo
      (read-string (concat prompt " : "))
;      (read-from-minibuffer (concat prompt " : "))
      )))
;
;(defun coq-build-command-from-db (db prompt &optional preformatquery)
;  "Ask for a keyword, with completion on keyword database DB and send to coq.
;See `coq-syntax-db' for DB structure."
;  (let* ((query (completing-read (concat prompt " (TAB for completion): ")
;				 db nil nil))
;	 (infos (cddr (assoc query db)))
;	 (s (car infos))
;	 ; default format is add a space at the end of query, for arguments.
;	 (preformat (or preformatquery '(lambda (s) (concat s " "))))
;	 (initialvalue (funcall preformat query))
;	 (whole-query
;	  (minibuffer-with-setup-hook
;	      'coq-move-cursor-at-sharp
;	    (read-string (concat prompt " : ") initialvalue t nil))))
;    whole-query))
;



(defun coq-build-regexp-list-from-db (db &optional filter)
  "Take a keyword database DB and return the list of regexps for font-lock.
If non-nil Optional argument FILTER is a function applying to each line of DB.
For each line if FILTER returns nil, then the keyword is not added to the
regexp.  See `coq-syntax-db' for DB structure."
  (let ((l db) res)
    (while l
      (let* ((hd (car l)) (tl (cdr l))	; hd is the first infos list
	     (color (nth 4 hd)))	; colorization string
; da: do this below, otherwise we get empty words in result!!
;	     (color (concat "\\_<" (nth 4 hd) "\\_>")))       ; colorization string
	;; TODO delete doublons
	(when (and color (or (not filter) (funcall filter hd)))
	  (setq res
		(nconc res (list
			    (concat "\\_<" color "\\_>")))))
	(setq l tl)))
    res))

(defun coq-build-opt-regexp-from-db (db &optional filter)
  "Take a keyword database DB and return a regexps for font-lock.
If non-nil optional argument FILTER is a function applying to each line of DB.
For each line if FILTER returns nil, then the keyword is not added to the
regexp.  See `coq-syntax-db' for DB structure."
  (let ((l db) res)
    (while l
      (let* ((hd (car l)) (tl (cdr l))	; hd is the first infos list
	     (color (nth 4 hd)))       ; colorization string
	;; TODO delete doublons
	(when (and color (or (not filter) (funcall filter hd)))
	  (setq res (nconc res (list color)))) ; careful: nconc destructive!
	(setq l tl)))
; da: next call is wrong?
;    (proof-ids-to-regexp res)))
    (concat "\\_<\\(?:" (mapconcat #'identity res "\\|") "\\)\\_>")))


;; Computes the max length of strings in a list
(defun max-length-db (db)
  "Return the length of the longest first element (menu label) of DB.
See `coq-syntax-db' for DB structure."
  (let ((l db) (res 0))
    (while l
      (let ((lgth (length (car (car l)))))
	(setq res (max lgth res))
	(setq l (cdr l))))
    res))



(defun coq-build-menu-from-db-internal (db lgth menuwidth)
  "Take a keyword database DB and return one insertion submenu.
Argument LGTH is the max size of the submenu.  Argument MENUWIDTH is the width
of the largest line in the menu (without abbrev and shortcut specifications).
Used by `coq-build-menu-from-db', which you should probably use instead.  See
`coq-syntax-db' for DB structure."
  (let ((l db) (res ()) (size lgth)
	(keybind-abbrev (if coq-use-yasnippet (substitute-command-keys " \\[yas-expand]")
                          (substitute-command-keys " \\[expand-abbrev]"))))
    (while (and l (> size 0))
      (let* ((hd (pop l))
	     (menu     	 (nth 0 hd)) ; e1 = menu entry
	     (abbrev   	 (nth 1 hd)) ; e2 = abbreviation
	     (complt   	 (nth 2 hd)) ; e3 = completion
	     ;; (state   (nth 3 hd)) ; e4 = state changing
	     ;; (color   (nth 4 hd)) ; e5 = colorization string
	     ;; (insertfn (nth 5 hd)) ; e6 = function for smart insertion not used
	     (menuhide 	 (nth 6 hd)) ; e7 = if non-nil : hide in menu
	     (entry-with (max (- menuwidth (length menu)) 0))
	     (spaces (make-string entry-with ? ))
	     ;;(restofmenu (coq-build-menu-from-db-internal tl (- size 1) menuwidth))
	     )
	(unless menuhide
	  (let ((menu-entry
		 (vector
		  ;; menu entry label
		  (concat menu
			  (if (not abbrev) ""
			    (concat spaces "(" abbrev keybind-abbrev ")")))
                  `(coq-insert-template ,complt)
		  t)))
	    (push menu-entry res)))
	(cl-decf size)))
    (nreverse res)))


(defun coq-build-title-menu (db size)
  "Build a title for the first submenu of DB, of size SIZE.
Return the string made of the first and the SIZE nth first element of DB,
separated by \"...\".  Used by `coq-build-menu-from-db'.  See `coq-syntax-db'
for DB structure."
    (concat (car-safe (car-safe db))
	  " ... "
	  (car-safe (car-safe (nthcdr (- size 1) db)))))

(defun coq-sort-menu-entries (menu)
  (sort menu
	#'(lambda (x y) (string<
			 (downcase (elt x 0))
			 (downcase (elt y 0))))))

(defun coq-build-menu-from-db (db &optional size)
  "Take a keyword database DB and return a list of insertion menus for them.
Submenus contain SIZE entries (default 30).  See `coq-syntax-db' for DB
structure."
  ;; sort is destructive for the list, so copy list before sorting
  (let* ((l (coq-sort-menu-entries (copy-sequence db))) (res ())
	 (wdth (+ 2 (max-length-db db)))
	 (sz (or size 30)) (lgth (length l)))
    (while l
      (if (<= lgth sz)
	  (setq res ;; careful: nconc destructive!
		(nconc res (list (cons
				  (coq-build-title-menu l lgth)
				  (coq-build-menu-from-db-internal l lgth wdth)))))
	(setq res ; careful: nconc destructive!
	      (nconc res (list (cons
				(coq-build-title-menu l sz)
				(coq-build-menu-from-db-internal l sz wdth))))))
      (setq l (nthcdr sz l))
      (setq lgth (length l)))
    res))

(defun coq-simple-abbrev-from-db (s)
  "Returns the coq db abbrevitaion string S cleaned from template markups."
  (if s (replace-regexp-in-string "#\\|@{\\(?1:[^{}]*\\)}" "" s)))

(defun coq-build-abbrev-table-from-db (db)
  "Take a keyword database DB and return an abbrev table.
See `coq-syntax-db' for DB structure."
  (let ((l db) (res ()))
    (while l
      (let* ((hd (car l))(tl (cdr l))	; hd is a list of length 3 or 4
	     (_e1 (car hd)) (tl1 (cdr hd)) ; e1 = menu entry
	     (e2 (car tl1)) (tl2 (cdr tl1)) ; e2 = abbreviation
	     (e3 (car tl2)) (_tl3 (cdr tl2)) ; e3 = completion
             (e3clean (coq-simple-abbrev-from-db e3)))
	;; careful: nconc destructive!
	(when e2 (push `(,e2 ,e3clean nil :system t) res))
	(setq l tl)))
    (nreverse res)))


(defun coq-db--replace-abbrev-yas (i mtch)
  "Translationi of a coq db abbrev spec to a yasnippet one."
  (cond
   ;; FIXME: define a "end point for the snippet: that is the place where the
   ;; cursor should go after fiolling the holes. Generally it is the place where
   ;; you want to continue editing.
   ((string-equal mtch "#") (concat "$" (int-to-string i)))
   (t (concat "${" (int-to-string i) ":" (match-string 1 mtch) "}"))
   )
  )

(defun coq-yas-snippet-from-db (s)
  "Translate the coq db abbrevitaion string S into the yasnippet abbrev format."
  (if s
    (let ((cpt 0))
      (replace-regexp-in-string
       "#\\|@{\\(?1:[^{}]*\\)}"
       (lambda (mtch) (setq cpt (+ 1 cpt)) (coq-db--replace-abbrev-yas cpt mtch))
       s))))

(defun coq-yas-snippet-table-from-db (db)
  "Take a keyword database DB and return a list of yasnippets.

The list contains lists or args to be given to yas-define-snippets.
See `coq-syntax-db' for DB structure."
  (let ((l db) (res ()))
    (while l
      (let* ((hd (car l))(tl (cdr l))	; hd is a list of length 3 or 4
	     (e1 (car hd)) (tl1 (cdr hd)) ; e1 = menu entry
	     (e2 (car tl1)) (tl2 (cdr tl1)) ; e2 = abbreviation
	     (e3 (car tl2)) (_tl3 (cdr tl2)) ; e3 = completion
             (yas_e3 (coq-yas-snippet-from-db e3)))
	;; careful: nconc destructive!
	(when e2 (push `(,e2 ,yas_e3 ,e1) res))
	(setq l tl)))
    (nreverse res)))


(defun filter-state-preserving (l)
  ; checkdoc-params: (l)
  "Not documented."
  (not (nth 3 l))) ; fourth argument is nil --> state preserving command

(defun filter-state-changing (l)
  ; checkdoc-params: (l)
  "Not documented."
  (nth 3 l)) ; fourth argument is nil --> state preserving command


;;A new face for tactics which fail when they don't kill the current goal
(defface coq-solve-tactics-face
  `((((background light)) :foreground "red")
    (((background dark)) :foreground "red1")
    ()) ; pour le noir et blanc
  "Face for names of closing tactics in proof scripts."
  :group 'proof-faces)

;;A face for cheating tactics
;; We use :box in addition to :background because box remains visible in
;; locked-region. :reverse-video is another solution.
(defface coq-cheat-face
  '(;(((class color) (background light)) . (:inverse-video t :foreground "red" :background "black"))
    ;(((class color) (background dark)) . (:inverse-video t :foreground "red1"))
    (((class color) (background light)) . (:box (:line-width -1 :color "red" :style nil) :background "red"))
    (((class color) (background dark)) . (:box (:line-width -1 :color "red1" :style nil) :background "red1"))
    (t . ())) ; monocolor or greyscale: no highlight
  "Face for names of cheating tactics in proof scripts."
  :group 'proof-faces)

(defface coq-symbol-binder-face
  '((t :inherit font-lock-type-face :bold coq-bold-unicode-binders))
  "Face for unicode binders, by default a bold version of `font-lock-type-face'."
  :group 'proof-faces)

(defface coq-symbol-face
  '((t :inherit font-lock-type-face :bold coq-bold-unicode-binders))
  "Face for unicode binders, by default a bold version of `font-lock-type-face'."
  :group 'proof-faces)

(defface coq-question-mark-face
  '((t :inherit font-lock-variable-name-face))
  "Face for Ltac binders and evars."
  :group 'proof-faces)

(defface coq-context-qualifier-face
  '((t :inherit font-lock-preprocessor-face :weight bold))
  "Face used for `ltac:', `constr:', and `uconstr:' headers."
  :group 'proof-faces)

;; This is so quite ugly, better have our own face.
;; (defface coq-button-face
;;   '((t :inherit custom-button :background "dark gray"
;;        ;; -1 avoids messing with line and column sizes
;;        :box (:line-width (-1 . -1) :style key-pressed)))
;;   ""
;;   :group 'proof-faces)

(defface coq-button-face
  '(
    (((class color) (background light)) . (:background "light gray"))
    (((class color) (background dark)) . (:background "gray" :foreground "black")))
  "Coq face for the (un)folding hypothesis button"
  :group 'proof-faces)

(defface coq-button-face-pressed
  '((((class color) (background light)) . (:background "gray"))
    (((class color) (background dark)) . (:background "dim gray" :foreground "black")))
  "Coq face for the (un)folding hypothesis button"
  :group 'proof-faces)

(defface coq-button-face-active
  '((((class color) (background light)) . (:background "black" :foreground "white"))
    (((class color) (background dark)) . (:background "grey24" :foreground "black")))
  "Coq face for the (un)folding hypothesis button"
  :group 'proof-faces)

(defconst coq-solve-tactics-face 'coq-solve-tactics-face
  "Expression that evaluates to a face.
Required so that \\='coq-solve-tactics-face is a proper facename")

(defconst coq-cheat-face 'coq-cheat-face
  "Expression that evaluates to a face.
Required so that \\='coq-cheat-face is a proper facename")

(defconst coq-symbol-binder-face 'coq-symbol-binder-face
  "Expression that evaluates to a face.
Required so that \\='coq-symbol-binder-face is a proper facename")

(defconst coq-symbol-face 'coq-symbol-face
  "Expression that evaluates to a face.
Required so that \\='coq-symbol-binder-face is a proper facename")

(defconst coq-question-mark-face 'coq-question-mark-face)


(defface coq-diffs-added-face
  '((t . (:inherit diff-refine-added)))
  "Face used to highlight added text in diffs"
  :group 'proof-faces)

(defface coq-diffs-removed-face
  '((t . (:inherit diff-refine-removed)))
  "Face used to highlight removed text in diffs"
  :group 'proof-faces)

(defface coq-diffs-added-bg-face
  '((t . (:inherit diff-added)))
  "Face used to highlight unchanged text in lines showing added text in diffs"
  :group 'proof-faces)

(defface coq-diffs-removed-bg-face
  '((t . (:inherit diff-removed)))
  "Face used to highlight unchanged text in lines showing removed text in diffs"
  :group 'proof-faces)

(defvar coq-tag-map
  '(("diff.added" . coq-diffs-added-face)
   ("diff.removed" . coq-diffs-removed-face)
   ("diff.added.bg" . coq-diffs-added-bg-face)
   ("diff.removed.bg" . coq-diffs-removed-bg-face)))

(provide 'coq-db)

;;; coq-db.el ends here

;** Local Variables: ***
;** fill-column: 80 ***
;** End: ***
